import json

from django.http import JsonResponse
from django.shortcuts import render, HttpResponse

# Archivos
from django.core.files.storage import FileSystemStorage
# Configuracion
from django.conf import settings

# Serializers json
from django.core import serializers

# Http Status
from http import HTTPStatus

# Modelo de colaborador
from colaboradores.models import Colaboradores

# Modelo cuenta de cliente
from clientes.models import CuentaCliente
from django.views.decorators.csrf import csrf_exempt

# Modelo de asistencia
from .models import AsistenciaCol

# Modelo Query
from django.db.models import Q

# Static Full Path
from utils.static_constants import STATIC_FULL_URL_PATH

# Datetime
from datetime import datetime, date

# Vistas para 'asistencias'

# --- Codigos de error ---
WITHOUT_EXIT = "WITHOUT_EXIT"
PENDING_EXIT = "PENDING_EXIT"
COL_NOT_FOUND = "COL_NOT_FOUND"

# --- Utilidades --
def obtener_col_por_curp_o_id(col_curp_or_id):
    col = None
    # Obtiene el posible colaborador mediante su CURP o ID
    curp_query = Q(col_curp=col_curp_or_id)
    if col_curp_or_id.isdigit():
        id_query = Q(col_id=col_curp_or_id)
        
    # Obtiene posible colaborador mediante CURP
    col = Colaboradores.objects.filter(curp_query).first()
        
    # Si el colaborador es nulo, lo intentara obtener mediante su ID
    if col is None and col_curp_or_id.isdigit():
        col = Colaboradores.objects.filter(id_query).first()
        
    return col

def get_col_data(request):
    if request.method == 'GET':
        # Response
        response = {
            "foto_perfil_url": "",
            "num_colaborador": "",
            "nombre": "",
            "apellido_paterno": "",
            "apellido_materno": "",
            "curp": "",
            "edad": "",

            "num_cuip": "",
            "nombre_jefe_inmediato": "",
            "turno": "",
            "nombre_oficina": "",
            "nombre_servicio": "",
            "nombre_sucursal": "",
            "puesto": ""
        }

        # Obtiene parametros de peticion
        curp_o_id = request.GET.get("curpOrId")

        col_query = obtener_col_por_curp_o_id(curp_o_id)
        """
        # Obtiene el colaborador correspondiente mediante la curp o id
        curp_query = Q(col_curp=curp_o_id)
        id_query = Q(col_id=curp_o_id)

        # Primero intenta obtenerlo con la curp
        col_query = Colaboradores.objects.filter(curp_query).first()

        # No encontro con curp, probara con id
        if col_query is None:
            col_query = Colaboradores.objects.filter(id_query).first()
        """

        # No encontro ninguno, devuelve error
        if col_query is None:
            # "No se encontro ningun colaborador mediante el id o curp"
            return HttpResponse(COL_NOT_FOUND,status=HTTPStatus.INTERNAL_SERVER_ERROR)

        # Obtiene los datos del colaborador
        response["foto_perfil_url"] = f"{STATIC_FULL_URL_PATH}/{col_query.col_foto_perfil}"
        response["num_colaborador"] = col_query.col_num_col
        response["nombre"] = f"{col_query.col_primer_nombre} {col_query.col_segundo_nombre}"
        response["apellido_paterno"] = col_query.col_apelli_paterno
        response["apellido_materno"] = col_query.col_apelli_materno
        response["curp"] = col_query.col_curp
        response["edad"] = col_query.col_edad

        datos_empleo = col_query.col_datos_empleo
        response["num_cuip"] = datos_empleo.num_cuip
        response["nombre_jefe_inmediato"] = datos_empleo.nombre_jefe_inmediato
        response["turno"] = datos_empleo.turno
        response["nombre_oficina"] = datos_empleo.oficina.oficina_nombre if datos_empleo.oficina is not None else ""
        response["nombre_servicio"] = datos_empleo.servicio.cli_nombre_cliente if datos_empleo.servicio is not None else ""
        response["nombre_sucursal"] = datos_empleo.sucursal.surc_nombre if datos_empleo.sucursal is not None else ""
        response["puesto"] = datos_empleo.puesto.puesto_nombre

        return JsonResponse(response)


    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)
    
def check_pending_exit(request):
    if request.method == "GET":
        response = False

        # Obtiene CURP o ID de colaboardor
        col_curp_or_id = request.GET.get("col_curp_or_id","")
        
        col = obtener_col_por_curp_o_id(col_curp_or_id)
        
        
        # Si no obtuvo ningun colaborador, devolvera un error
        if col is None:
            return HttpResponse(COL_NOT_FOUND,status=500)
            
        
        # -- Verifica si el colaborador tiene una salida pendiente, de ser asi, retornara un error --
        asistencia_curp_verify_query = Q(curp_col=col.col_curp)
        asistencia_curp_verify_query.add(Q(fecha_hora_salida=None), Q.AND)
        asistencia_curp_verify_query.add(~Q(fecha_hora_entrada=None), Q.AND)
        asistencia_curp_verify = AsistenciaCol.objects.filter(asistencia_curp_verify_query).first()
        
        if asistencia_curp_verify is not None:
            response = True
            
            
        return HttpResponse(response)
        
        
    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

@csrf_exempt
def register_assistance(request):
    if request.method == "POST":
        response = {
            "success": False,
            "error": False,
            "error_code": ""
        }

        # Obtiene los el cuerpo de la peticion
        body = json.loads(request.POST["body"])
        # Obtiene la imagen final de la asistencia
        image = request.FILES.get("imgAsistencia")

        # Obtiene los datos individuales del cuerpo de la petición
        client_account_email = body["clientAccountEmail"]
        col_curp = body["colCurp"]
        col_shift = body["colShift"]
        assistance_local_datetime = body["localDatetime"]
        assistance_local_datetime = datetime.strptime(assistance_local_datetime, "%d-%m-%Y %H:%M:%S")

        # Obtiene cuenta de cliente correspondiente
        cuenta_cliente = CuentaCliente.objects.get(correo=client_account_email)

        # Obtiene colaborador correspondiente
        col = Colaboradores.objects.get(col_curp=col_curp)
        
        # -- Verifica si el colaborador tiene una salida pendiente, de ser asi, retornara un error --
        asistencia_verify_query = Q(curp_col=col_curp)
        asistencia_verify_query.add(Q(fecha_hora_salida=None), Q.AND)
        asistencia_verify_query.add(~Q(fecha_hora_entrada=None), Q.AND)
        asistencia_verify = AsistenciaCol.objects.filter(asistencia_verify_query).first()
        
        if asistencia_verify is not None:
            response["success"] = False
            response["error"] = True
            response["error_code"] = PENDING_EXIT
            
            return JsonResponse(response)


        # Sube imagen de asistencia al servidor
        ruta_imagen_asistencia = ""
        if image:
            fs = FileSystemStorage(
                location=f"{settings.MEDIA_ROOT}/images/col-asistencias/"
            )
            filename = image.name
            fs.save(filename, image)
            ruta_imagen_asistencia = f"media/images/col-asistencias/{filename}"

        # Inserta registro de asistencia
        asistencia_col = AsistenciaCol()
        asistencia_col.primer_nombre_col = col.col_primer_nombre
        asistencia_col.segundo_nombre_col = col.col_segundo_nombre.strip()
        asistencia_col.apellido_paterno_col = col.col_apelli_paterno
        asistencia_col.apellido_materno_col = col.col_apelli_materno
        asistencia_col.curp_col = col_curp
        asistencia_col.num_col = col.col_num_col
        asistencia_col.puesto_col = col.col_datos_empleo.puesto.puesto_nombre
        asistencia_col.nombre_oficina_col = col.col_datos_empleo.oficina.oficina_nombre
        asistencia_col.turno_col = col_shift
        asistencia_col.fecha_hora_entrada = assistance_local_datetime

        asistencia_col.nombre_servicio = cuenta_cliente.cliente.cli_nombre_cliente
        asistencia_col.nombre_sucursal = cuenta_cliente.sucursal.surc_nombre if cuenta_cliente.tiene_sucursal() else ""

        asistencia_col.id_servicio = cuenta_cliente.cliente.cli_id
        asistencia_col.id_sucursal = cuenta_cliente.sucursal.surc_id if cuenta_cliente.tiene_sucursal() else None

        asistencia_col.ruta_archivo_asistencia = ruta_imagen_asistencia

        asistencia_col.save()

        response["success"] = True


        return JsonResponse(
            response
        )


    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)
    
    
    
@csrf_exempt
def register_assistance_exit(request):
    if request.method == "POST":
        response = {
            "success": False,
            "error": False,
            "error_code": ""
        }
        
        # Obtiene los el cuerpo de la peticion
        body = json.loads(request.body)
        
        # Obtiene la CURP o ID del colaborador
        col_curp_or_id = body["colCurpOrId"]
        
        # Obtiene el posible colaborador por curp o id
        col = obtener_col_por_curp_o_id(col_curp_or_id)
        
        
        if col is None:
            response["error"] = True
            response["error_code"] = COL_NOT_FOUND
            return JsonResponse(response)
            
        col_curp = col.col_curp
        
        # Verifica si existe un registro de asistencia que tenga una salida pendiente
        asistencia_col_query = Q(curp_col=col_curp)
        asistencia_col_query.add(Q(fecha_hora_salida=None), Q.AND)
        asistencia_col_query.add(~Q(fecha_hora_entrada=None), Q.AND)
        
        asistencia_col = AsistenciaCol.objects.filter(asistencia_col_query).first()
        
        # Si actualmente no tiene realmente una salida pendiente retornara un error
        if asistencia_col is None:
            response["error"] = True
            response["error_code"] = WITHOUT_EXIT
            return JsonResponse(response)
        
        # Si si tiene una salida pendiente, simplemente registrara la fecha y hora de salida correspondiente
        asistencia_col.fecha_hora_salida = datetime.now()
        asistencia_col.save()
        
        response["success"] = True
        return JsonResponse(response)
        
        
        
    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)
    
    
    
    
    
    